<?php
/**
 * Plugin Name: Pingback Source Reveal
 * Description: Advanced pingback management with rich metadata display. Shows real origin website (final URL, site name, page title, anchor text) plus IP/rDNS/UA and full moderation controls. Perfect for bloggers and content creators.
 * Version: 1.0.0
 * Author: NetClick
 * License: GPL-2.0+
 * Text Domain: pingback-source-reveal
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.8
 * Requires PHP: 7.4
 */

if (!defined('ABSPATH')) exit;

define('PBR_SLUG', 'pingback-source-reveal');
define('PBR_VER',  '1.0.0');

/* ============================================================================
 * Capture basic source data on insert (pingbacks/trackbacks only)
 * ==========================================================================*/
add_action('comment_post', function ($comment_ID, $approved, $commentdata) {
    $c = get_comment($comment_ID);
    if (!$c || !in_array($c->comment_type, ['pingback','trackback'], true)) return;

    // Prefer comment author URL; fallback to first URL in content
    $src = $c->comment_author_url;
    if (!$src && preg_match('/https?:\/\/[^\s)"]+/i', $c->comment_content, $m)) $src = $m[0];

    if ($src) {
        add_comment_meta($comment_ID, 'pb_source_url', esc_url_raw($src), true);
        $host = wp_parse_url($src, PHP_URL_HOST);
        if ($host) add_comment_meta($comment_ID, 'pb_source_host', sanitize_text_field($host), true);
    }
}, 10, 3);

/* ============================================================================
 * Admin menu + Admin Bar
 * ==========================================================================*/
add_action('admin_menu', function () {
    add_menu_page(
        __('Pingbacks','pbr'),
        __('Pingbacks','pbr'),
        'moderate_comments',
        PBR_SLUG,
        'pbr_render_admin_page',
        'dashicons-admin-links',
        60
    );
    
    // Add PRO upgrade submenu - Shows in-admin upgrade page
    add_submenu_page(
        PBR_SLUG,
        __('Upgrade to PRO','pbr'),
        __('🚀 Upgrade to PRO','pbr'),
        'moderate_comments',
        'pingback-pro-upgrade',
        'pbr_render_upgrade_page'
    );
    
    // Add direct redirect to external landing page
    add_submenu_page(
        PBR_SLUG,
        __('Get PRO Version','pbr'),
        __('⭐ Get PRO Version','pbr'),
        'moderate_comments',
        'pingback-get-pro',
        function() {
            wp_redirect('https://netclick.tech/pro');
            exit;
        }
    );
});

add_action('admin_bar_menu', function ($bar) {
    if (!current_user_can('moderate_comments') || !is_admin_bar_showing()) return;
    $bar->add_node([
        'id'    => PBR_SLUG,
        'title' => __('Pingbacks','pbr'),
        'href'  => admin_url('admin.php?page='.PBR_SLUG),
        'meta'  => ['title'=>__('View Pingbacks','pbr')]
    ]);
    
    // Add PRO upgrade node
    $bar->add_node([
        'id'    => PBR_SLUG . '-pro',
        'parent'=> PBR_SLUG,
        'title' => __('🚀 Upgrade to PRO','pbr'),
        'href'  => 'https://netclick.tech/pro',
        'meta'  => ['title'=>__('Upgrade to PingBack Source Reveal PRO','pbr')]
    ]);
}, 90);

/* ============================================================================
 * Helpers
 * ==========================================================================*/
function pbr_is_pingtype($c){ return $c && in_array($c->comment_type, ['pingback','trackback'], true); }
function pbr_status_label($c){
    if ($c->comment_approved === '1') return 'approved';
    if ($c->comment_approved === '0') return 'pending';
    return $c->comment_approved; // 'spam' | 'trash'
}
function pbr_nonce($action){ return wp_create_nonce('pbr_'.$action); }
function pbr_action_url($action, $id){
    return wp_nonce_url(admin_url('admin-post.php?action=pbr_'.$action.'&c='.$id), 'pbr_'.$action);
}
function pbr_registrable_domain($host){
    if (!$host) return '';
    $host = strtolower($host);
    $parts = array_reverse(explode('.', $host));
    if (count($parts) >= 2) return $parts[1].'.'.$parts[0];
    return $host;
}

/* ============================================================================
 * Metadata resolve + fetch
 * ==========================================================================*/
function pbr_effective_url_from_response($url, $response){
    // Try to discover final effective URL after redirects
    if (is_array($response) && isset($response['http_response']) && is_object($response['http_response'])) {
        $obj = $response['http_response']->get_response_object();
        if (is_object($obj) && !empty($obj->url)) return $obj->url;
    }
    $loc = wp_remote_retrieve_header($response, 'location');
    if (!empty($loc)) return $loc;
    return $url;
}

function pbr_follow_redirects($url){
    // Try HEAD first (fast), then fallback to GET if HEAD unsupported
    $ua = 'WordPress/'.get_bloginfo('version').' (+'.home_url().') Pingback-Verify';
    $args = ['timeout'=>10, 'redirection'=>5, 'user-agent'=>$ua, 'reject_unsafe_urls'=>true];
    $head = wp_remote_head($url, $args);
    if (!is_wp_error($head)) {
        $code  = intval(wp_remote_retrieve_response_code($head));
        $final = pbr_effective_url_from_response($url, $head);
        return [$final, $code];
    }
    $get = wp_remote_get($url, $args + ['headers'=>['Accept'=>'text/html']]);
    if (is_wp_error($get)) return [$url, 0];
    $code  = intval(wp_remote_retrieve_response_code($get));
    $final = pbr_effective_url_from_response($url, $get);
    return [$final, $code];
}

function pbr_fetch_meta($url, $your_domain){
    $ua = 'WordPress/'.get_bloginfo('version').' (+'.home_url().') Pingback-Verify';
    $res = wp_remote_get($url, [
        'timeout'            => 12,
        'redirection'        => 5,
        'user-agent'         => $ua,
        'reject_unsafe_urls' => true,
        'headers'            => ['Accept'=>'text/html']
    ]);
    if (is_wp_error($res)) return ['status'=>0];
    $status = intval(wp_remote_retrieve_response_code($res));
    $body   = wp_remote_retrieve_body($res);
    if (!$body) return ['status'=>$status];

    // Trim body to first 512KB to avoid memory issues
    if (strlen($body) > 524288) $body = substr($body, 0, 524288);

    // <title>
    $title = '';
    if (preg_match('/<title[^>]*>(.*?)<\/title>/is', $body, $m)) $title = wp_strip_all_tags($m[1]);

    // og:site_name
    $site_name = '';
    if (preg_match('/<meta[^>]+property=["\']og:site_name["\'][^>]+content=["\']([^"\']+)["\']/i', $body, $m)) {
        $site_name = wp_strip_all_tags($m[1]);
    }

    // Try <meta name="application-name"> or <meta property="og:site" ...> as fallback
    if (!$site_name && preg_match('/<meta[^>]+name=["\']application-name["\'][^>]+content=["\']([^"\']+)["\']/i', $body, $m)) {
        $site_name = wp_strip_all_tags($m[1]);
    }

    // Anchor text for links to our site
    $anchor = '';
    $link_present = 0;
    $pattern = '~<a\b[^>]*href=["\']([^"\']+)["\'][^>]*>(.*?)</a>~is';
    if (preg_match_all($pattern, $body, $all, PREG_SET_ORDER)) {
        foreach ($all as $a) {
            $href = html_entity_decode($a[1], ENT_QUOTES);
            if (stripos($href, $your_domain) !== false) {
                $link_present = 1;
                $anchor = trim(wp_strip_all_tags($a[2]));
                if ($anchor !== '') break;
            }
        }
    }

    return [
        'status'       => $status,
        'title'        => $title,
        'site_name'    => $site_name,
        'link_present' => $link_present,
        'anchor'       => $anchor,
    ];
}

function pbr_refresh_meta_for_comment($comment_id){
    $c = get_comment($comment_id);
    if (!$c || !pbr_is_pingtype($c)) return;

    $src = get_comment_meta($comment_id, 'pb_source_url', true);
    if (!$src) $src = $c->comment_author_url;
    if (!$src) return;

    list($final, $status) = pbr_follow_redirects($src);
    if ($final) update_comment_meta($comment_id, 'pb_final_url', esc_url_raw($final));
    if ($status) update_comment_meta($comment_id, 'pb_http_status', intval($status));

    $host = wp_parse_url($final ?: $src, PHP_URL_HOST);
    if ($host) {
        update_comment_meta($comment_id, 'pb_source_host', sanitize_text_field($host));
        update_comment_meta($comment_id, 'pb_root_domain', pbr_registrable_domain($host));
    }

    $your_domain = wp_parse_url(home_url(), PHP_URL_HOST);
    $meta = pbr_fetch_meta($final ?: $src, $your_domain);
    foreach ([
        'title'        => 'pb_page_title',
        'site_name'    => 'pb_site_name',
        'anchor'       => 'pb_anchor_text',
        'status'       => 'pb_http_status',
        'link_present' => 'pb_link_present',
    ] as $k=>$mk) {
        if (isset($meta[$k])) update_comment_meta($comment_id, $mk, is_scalar($meta[$k]) ? $meta[$k] : json_encode($meta[$k]));
    }
}

/* ============================================================================
 * Single + Bulk admin actions
 * ==========================================================================*/
add_action('admin_post_pbr_refreshmeta', function(){
    if (!current_user_can('moderate_comments')) wp_die(__('Access denied.'));
    check_admin_referer('pbr_refreshmeta');
    $id = isset($_GET['c']) ? intval($_GET['c']) : 0;
    if ($id) pbr_refresh_meta_for_comment($id);
    wp_safe_redirect(wp_get_referer() ?: admin_url('admin.php?page='.PBR_SLUG));
    exit;
});

add_action('admin_post_pbr_bulk', function () {
    if (!current_user_can('moderate_comments')) wp_die(__('Access denied.'));
    check_admin_referer('pbr_bulk');

    $ids = isset($_POST['comment_ids']) ? array_map('intval', (array)$_POST['comment_ids']) : [];
    $op  = isset($_POST['bulk_op']) ? sanitize_text_field($_POST['bulk_op']) : '';

    foreach ($ids as $id) {
        switch ($op) {
            case 'approve':   wp_set_comment_status($id, 'approve'); break;
            case 'unapprove': wp_set_comment_status($id, 'hold');    break;
            case 'spam':      wp_spam_comment($id);                  break;
            case 'trash':     wp_trash_comment($id);                 break;
            case 'delete':    wp_delete_comment($id, true);          break;
            case 'verify':    pbr_refresh_meta_for_comment($id);     break;
        }
    }
    wp_safe_redirect(wp_get_referer() ?: admin_url('admin.php?page='.PBR_SLUG));
    exit;
});

foreach (['approve','unapprove','spam','trash','untrash','delete'] as $op) {
    add_action('admin_post_pbr_'.$op, function() use ($op){
        if (!current_user_can('moderate_comments')) wp_die(__('Access denied.'));
        check_admin_referer('pbr_'.$op);
        $id = isset($_GET['c']) ? intval($_GET['c']) : 0;
        if ($id) {
            switch ($op) {
                case 'approve':   wp_set_comment_status($id, 'approve'); break;
                case 'unapprove': wp_set_comment_status($id, 'hold');    break;
                case 'spam':      wp_spam_comment($id);                  break;
                case 'trash':     wp_trash_comment($id);                 break;
                case 'untrash':   wp_untrash_comment($id);               break;
                case 'delete':    wp_delete_comment($id, true);          break;
            }
        }
        wp_safe_redirect(wp_get_referer() ?: admin_url('admin.php?page='.PBR_SLUG));
        exit;
    });
}

/* ============================================================================
 * Admin Page
 * ==========================================================================*/
function pbr_render_admin_page() {
    if (!current_user_can('moderate_comments')) wp_die(__('Access denied.'));

    $status   = isset($_GET['pbr_status']) ? sanitize_text_field($_GET['pbr_status']) : 'all';
    $s        = isset($_GET['s']) ? sanitize_text_field($_GET['s']) : '';
    $per_page = max(5, min(100, isset($_GET['per_page']) ? intval($_GET['per_page']) : 20));
    $paged    = max(1, isset($_GET['paged']) ? intval($_GET['paged']) : 1);
    $offset   = ($paged - 1) * $per_page;

    $args = [
        'type__in' => ['pingback','trackback'],
        'number'   => $per_page,
        'offset'   => $offset,
        'orderby'  => 'comment_date_gmt',
        'order'    => 'DESC',
        'status'   => 'all',
        'search'   => $s ? '*'.esc_attr($s).'*' : '',
    ];
    if     ($status === 'approved') $args['status'] = 'approve';
    elseif ($status === 'pending')  $args['status'] = 'hold';
    elseif ($status === 'spam')     $args['status'] = 'spam';
    elseif ($status === 'trash')    $args['status'] = 'trash';

    $comments    = get_comments($args);
    $total_found = get_comments(array_merge($args, ['count'=>true, 'number'=>0, 'offset'=>0]));
    $total_pages = max(1, ceil($total_found / $per_page));
    $base_url    = admin_url('admin.php?page='.PBR_SLUG);
    ?>
    <div class="wrap">
        <h1>
            <span style="color: #2563eb; font-weight: bold;">NetClick</span> - <?php esc_html_e('Pingbacks','pbr'); ?>
            <span style="font-size: 14px; font-weight: normal; color: #666; margin-left: 10px;">by NetClick</span>
        </h1>

        <!-- NetClick Branding Banner -->
        <div style="background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%); border: 1px solid #cbd5e1; border-radius: 6px; padding: 15px; margin-bottom: 20px;">
            <div style="display: flex; align-items: center; justify-content: space-between;">
                <div>
                    <strong style="color: #2563eb; font-size: 16px;">NetClick PingBack Source Reveal</strong>
                    <div style="color: #64748b; font-size: 13px; margin-top: 2px;">Basic Version - Track and manage pingbacks with ease</div>
                </div>
                <div style="text-align: right;">
                    <a href="https://netclick.tech/pro" target="_blank" style="
                        display: inline-block;
                        background: linear-gradient(135deg, #10b981 0%, #059669 100%);
                        color: white;
                        padding: 8px 16px;
                        text-decoration: none;
                        border-radius: 4px;
                        font-weight: 600;
                        font-size: 13px;
                        transition: all 0.3s ease;
                    " onmouseover="this.style.transform='translateY(-1px)'" onmouseout="this.style.transform='translateY(0)'">
                        🚀 Upgrade to PRO
                    </a>
                </div>
            </div>
        </div>

        <form method="get" style="margin:12px 0;">
            <input type="hidden" name="page" value="<?php echo esc_attr(PBR_SLUG); ?>" />
            <input type="search" name="s" value="<?php echo esc_attr($s); ?>" placeholder="<?php esc_attr_e('Search URL, title, content…','pbr'); ?>" />
            <select name="pbr_status">
                <?php foreach (['all'=>'All','approved'=>'Approved','pending'=>'Pending','spam'=>'Spam','trash'=>'Trash'] as $k=>$label)
                    printf('<option value="%s"%s>%s</option>', esc_attr($k), selected($status, $k, false), esc_html($label)); ?>
            </select>
            <label style="margin-left:8px;">
                <?php esc_html_e('Per page:','pbr'); ?>
                <input type="number" name="per_page" min="5" max="100" value="<?php echo esc_attr($per_page); ?>" style="width:70px;">
            </label>
            <button class="button button-primary"><?php esc_html_e('Filter','pbr'); ?></button>
        </form>

        <form method="post" action="<?php echo esc_url( admin_url('admin-post.php') ); ?>">
            <input type="hidden" name="action" value="pbr_bulk">
            <?php wp_nonce_field('pbr_bulk'); ?>

            <div class="tablenav top" style="margin:6px 0 10px;">
                <div class="alignleft actions bulkactions">
                    <select name="bulk_op">
                        <option value=""><?php esc_html_e('Bulk actions','pbr'); ?></option>
                        <option value="verify"><?php esc_html_e('Verify metadata','pbr'); ?></option>
                        <option value="approve"><?php esc_html_e('Approve','pbr'); ?></option>
                        <option value="unapprove"><?php esc_html_e('Unapprove','pbr'); ?></option>
                        <option value="spam"><?php esc_html_e('Mark as Spam','pbr'); ?></option>
                        <option value="trash"><?php esc_html_e('Move to Trash','pbr'); ?></option>
                        <option value="delete"><?php esc_html_e('Delete Permanently','pbr'); ?></option>
                    </select>
                    <button class="button action"><?php esc_html_e('Apply','pbr'); ?></button>
                </div>
                <div class="tablenav-pages">
                    <?php
                    echo paginate_links([
                        'base'      => add_query_arg(['page'=>PBR_SLUG,'pbr_status'=>$status,'s'=>$s,'per_page'=>$per_page,'paged'=>'%#%'], $base_url),
                        'format'    => '',
                        'prev_text' => '«',
                        'next_text' => '»',
                        'total'     => $total_pages,
                        'current'   => $paged,
                    ]);
                    ?>
                </div>
            </div>

            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <td class="manage-column check-column"><input type="checkbox" onclick="jQuery('input[name=&quot;comment_ids[]&quot;]').prop('checked', this.checked);" /></td>
                        <th><?php esc_html_e('Source','pbr'); ?></th>
                        <th><?php esc_html_e('Details','pbr'); ?></th>
                        <th><?php esc_html_e('Linked Post','pbr'); ?></th>
                        <th><?php esc_html_e('Date','pbr'); ?></th>
                        <th><?php esc_html_e('Status','pbr'); ?></th>
                        <th><?php esc_html_e('Actions','pbr'); ?></th>
                    </tr>
                </thead>
                <tbody>
                <?php if ($comments): foreach ($comments as $c):
                    if (!pbr_is_pingtype($c)) continue;

                    // URLs & meta
                    $src   = get_comment_meta($c->comment_ID,'pb_source_url',true) ?: $c->comment_author_url;
                    $final = get_comment_meta($c->comment_ID,'pb_final_url',true) ?: $src;
                    $host  = get_comment_meta($c->comment_ID,'pb_source_host',true) ?: ($final ? wp_parse_url($final, PHP_URL_HOST) : '');
                    $root  = get_comment_meta($c->comment_ID,'pb_root_domain',true) ?: ($host ? pbr_registrable_domain($host) : '');
                    $title = get_comment_meta($c->comment_ID,'pb_page_title',true);
                    $site  = get_comment_meta($c->comment_ID,'pb_site_name',true);
                    $anch  = get_comment_meta($c->comment_ID,'pb_anchor_text',true);
                    $http  = intval(get_comment_meta($c->comment_ID,'pb_http_status',true));
                    $linkok= intval(get_comment_meta($c->comment_ID,'pb_link_present',true));

                    // Post, IP, UA, rDNS
                    $post = get_post($c->comment_post_ID);
                    $ip   = $c->comment_author_IP;
                    $rdns = $ip ? @gethostbyaddr($ip) : '';
                    $ua   = $c->comment_agent;
                    $st   = pbr_status_label($c);
                    ?>
                    <tr>
                        <th class="check-column"><input type="checkbox" name="comment_ids[]" value="<?php echo esc_attr($c->comment_ID); ?>" /></th>
                        
                        <td>
                            <?php if ($final): ?>
                                <a href="<?php echo esc_url($final); ?>" target="_blank" rel="noopener">
                                    <?php echo esc_html($site ?: $root ?: $host ?: $final); ?>
                                </a>
                                <?php if ($title): ?>
                                    <div style="color:#666;font-size:11px;"><?php echo esc_html($title); ?></div>
                                <?php endif; ?>
                                <div style="color:#666;font-size:11px;word-break:break-all;"><?php echo esc_html($final); ?></div>
                                <?php if ($anch): ?>
                                    <div style="margin-top:2px;font-size:11px;"><strong><?php _e('Anchor:','pbr'); ?></strong> <?php echo esc_html($anch); ?></div>
                                <?php endif; ?>
                            <?php else: ?>
                                <em><?php esc_html_e('Unknown','pbr'); ?></em>
                            <?php endif; ?>
                        </td>

                        <td>
                            <?php if ($http): ?><div><code><?php echo esc_html($http); ?></code></div><?php endif; ?>
                            <?php if ($ip):   ?><div><code><?php echo esc_html($ip); ?></code></div><?php endif; ?>
                            <?php if ($rdns && $rdns!==$ip): ?><div style="color:#666;font-size:11px;"><?php echo esc_html($rdns); ?></div><?php endif; ?>
                            <?php if ($ua):   ?><div style="color:#666;font-size:11px;"><?php echo esc_html($ua); ?></div><?php endif; ?>
                            <div style="margin-top:4px;">
                                <a href="<?php echo esc_url( wp_nonce_url(admin_url('admin-post.php?action=pbr_refreshmeta&c='.$c->comment_ID),'pbr_refreshmeta') ); ?>">
                                    <?php _e('Verify / Refresh','pbr'); ?>
                                </a>
                                <?php if ($linkok && $http >= 200 && $http < 400): ?>
                                    <span style="color:#168f3a;font-size:11px;margin-left:6px;"><?php _e('[Backlink found]','pbr'); ?></span>
                                <?php elseif ($linkok === 0 && $http): ?>
                                    <span style="color:#a00;font-size:11px;margin-left:6px;"><?php _e('[No backlink detected]','pbr'); ?></span>
                                <?php endif; ?>
                            </div>
                        </td>

                        <td>
                            <?php if ($post): ?>
                                <a href="<?php echo esc_url(get_edit_post_link($post->ID)); ?>"><?php echo esc_html(get_the_title($post)); ?></a>
                                <div style="color:#666; font-size:11px;">
                                    <a href="<?php echo esc_url(get_permalink($post)); ?>" target="_blank" rel="noopener"><?php esc_html_e('View','pbr'); ?></a>
                                </div>
                            <?php endif; ?>
                        </td>

                        <td><?php echo esc_html( get_date_from_gmt($c->comment_date_gmt, 'Y-m-d H:i') ); ?></td>

                        <td><?php echo esc_html(ucfirst($st)); ?></td>

                        <td>
                            <?php if ($st !== 'approved'): ?>
                                <a href="<?php echo esc_url(pbr_action_url('approve', $c->comment_ID)); ?>"><?php esc_html_e('Approve','pbr'); ?></a> |
                            <?php else: ?>
                                <a href="<?php echo esc_url(pbr_action_url('unapprove', $c->comment_ID)); ?>"><?php esc_html_e('Unapprove','pbr'); ?></a> |
                            <?php endif; ?>
                            <?php if ($st !== 'spam'): ?>
                                <a href="<?php echo esc_url(pbr_action_url('spam', $c->comment_ID)); ?>"><?php esc_html_e('Spam','pbr'); ?></a> |
                            <?php endif; ?>
                            <?php if ($st !== 'trash'): ?>
                                <a href="<?php echo esc_url(pbr_action_url('trash', $c->comment_ID)); ?>"><?php esc_html_e('Trash','pbr'); ?></a> |
                            <?php else: ?>
                                <a href="<?php echo esc_url(pbr_action_url('untrash', $c->comment_ID)); ?>"><?php esc_html_e('Restore','pbr'); ?></a> |
                            <?php endif; ?>
                            <a href="<?php echo esc_url(pbr_action_url('delete', $c->comment_ID)); ?>" onclick="return confirm('<?php echo esc_js(__('Delete permanently?','pbr')); ?>');"><?php esc_html_e('Delete','pbr'); ?></a>
                        </td>
                    </tr>
                <?php endforeach; else: ?>
                    <tr><td colspan="7"><?php esc_html_e('No pingbacks found.','pbr'); ?></td></tr>
                <?php endif; ?>
                </tbody>
            </table>

            <div class="tablenav bottom" style="margin-top:8px;">
                <div class="alignleft actions bulkactions">
                    <select name="bulk_op">
                        <option value=""><?php esc_html_e('Bulk actions','pbr'); ?></option>
                        <option value="verify"><?php esc_html_e('Verify metadata','pbr'); ?></option>
                        <option value="approve"><?php esc_html_e('Approve','pbr'); ?></option>
                        <option value="unapprove"><?php esc_html_e('Unapprove','pbr'); ?></option>
                        <option value="spam"><?php esc_html_e('Mark as Spam','pbr'); ?></option>
                        <option value="trash"><?php esc_html_e('Move to Trash','pbr'); ?></option>
                        <option value="delete"><?php esc_html_e('Delete Permanently','pbr'); ?></option>
                    </select>
                    <button class="button action"><?php esc_html_e('Apply','pbr'); ?></button>
                </div>
                <div class="tablenav-pages">
                    <?php
                    echo paginate_links([
                        'base'      => add_query_arg(['page'=>PBR_SLUG,'pbr_status'=>$status,'s'=>$s,'per_page'=>$per_page,'paged'=>'%#%'], $base_url),
                        'format'    => '',
                        'prev_text' => '«',
                        'next_text' => '»',
                        'total'     => $total_pages,
                        'current'   => $paged,
                    ]);
                    ?>
                </div>
            </div>
        </form>

        <!-- Upgrade to PRO Section -->
        <div style="margin-top: 40px; padding: 25px; background: linear-gradient(135deg, #2563eb 0%, #1e40af 100%); border-radius: 8px; color: white;">
            <div style="display: flex; align-items: flex-start; justify-content: space-between; flex-wrap: wrap; gap: 20px;">
                <div style="flex: 1; min-width: 280px;">
                    <h2 style="margin: 0 0 12px 0; color: white; font-size: 22px; line-height: 1.3;">
                        <span style="background: #10b981; color: white; padding: 3px 6px; border-radius: 3px; font-size: 11px; margin-right: 6px; vertical-align: middle;">PRO</span>
                        Upgrade to PingBack Source Reveal PRO
                    </h2>
                    <p style="margin: 0 0 16px 0; opacity: 0.9; line-height: 1.5; font-size: 14px;">
                        Get advanced features like email alerts, auto-rules, analytics, CSV exports, and multi-channel notifications. 
                        Take your pingback management to the next level!
                    </p>
                    <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 8px 16px; margin-bottom: 0;">
                        <div style="display: flex; align-items: center; gap: 6px; font-size: 13px;">
                            <span style="color: #10b981; font-weight: bold;">✓</span>
                            <span>Email Alerts (HTML Templates)</span>
                        </div>
                        <div style="display: flex; align-items: center; gap: 6px; font-size: 13px;">
                            <span style="color: #10b981; font-weight: bold;">✓</span>
                            <span>Advanced CSV Export</span>
                        </div>
                        <div style="display: flex; align-items: center; gap: 6px; font-size: 13px;">
                            <span style="color: #10b981; font-weight: bold;">✓</span>
                            <span>Auto-Rules Engine</span>
                        </div>
                        <div style="display: flex; align-items: center; gap: 6px; font-size: 13px;">
                            <span style="color: #10b981; font-weight: bold;">✓</span>
                            <span>Multi-Channel Notifications</span>
                        </div>
                        <div style="display: flex; align-items: center; gap: 6px; font-size: 13px;">
                            <span style="color: #10b981; font-weight: bold;">✓</span>
                            <span>Analytics Dashboard</span>
                        </div>
                        <div style="display: flex; align-items: center; gap: 6px; font-size: 13px;">
                            <span style="color: #10b981; font-weight: bold;">✓</span>
                            <span>Security Features</span>
                        </div>
                    </div>
                </div>
                <div style="text-align: center; min-width: 180px; flex-shrink: 0;">
                    <div style="background: rgba(255,255,255,0.1); padding: 18px 16px; border-radius: 6px; margin-bottom: 12px;">
                        <div style="font-size: 28px; font-weight: bold; color: #10b981; line-height: 1;">$29</div>
                        <div style="font-size: 13px; opacity: 0.8; margin-top: 2px;">One-time payment</div>
                        <div style="font-size: 11px; opacity: 0.7; margin-top: 3px;">Lifetime updates</div>
                    </div>
                    <a href="https://netclick.tech/pro" target="_blank" style="
                        display: inline-block;
                        background: linear-gradient(135deg, #10b981 0%, #059669 100%);
                        color: white !important;
                        padding: 10px 20px;
                        text-decoration: none;
                        border-radius: 5px;
                        font-weight: 600;
                        font-size: 14px;
                        transition: all 0.3s ease;
                        box-shadow: 0 3px 12px rgba(16, 185, 129, 0.3);
                        white-space: nowrap;
                    " onmouseover="this.style.transform='translateY(-1px)';this.style.boxShadow='0 4px 15px rgba(16, 185, 129, 0.4)'" onmouseout="this.style.transform='translateY(0)';this.style.boxShadow='0 3px 12px rgba(16, 185, 129, 0.3)'">
                        🚀 Get PRO Now
                    </a>
                </div>
            </div>
        </div>
    </div>
    <?php
}

/* ============================================================================
 * Upgrade Page
 * ==========================================================================*/
function pbr_render_upgrade_page() {
    if (!current_user_can('moderate_comments')) wp_die(__('Access denied.'));
    ?>
    <div class="wrap">
        <h1>
            <span style="color: #2563eb; font-weight: bold;">NetClick</span> - <?php esc_html_e('Upgrade to PRO','pbr'); ?>
            <span style="font-size: 14px; font-weight: normal; color: #666; margin-left: 10px;">by NetClick</span>
        </h1>

        <!-- NetClick Branding Banner -->
        <div style="background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%); border: 1px solid #cbd5e1; border-radius: 6px; padding: 15px; margin-bottom: 20px;">
            <div style="display: flex; align-items: center; justify-content: space-between;">
                <div>
                    <strong style="color: #2563eb; font-size: 16px;">NetClick PingBack Source Reveal PRO</strong>
                    <div style="color: #64748b; font-size: 13px; margin-top: 2px;">Advanced pingback management with premium features</div>
                </div>
                <div style="text-align: right;">
                    <a href="https://netclick.tech/pro" target="_blank" style="
                        display: inline-block;
                        background: linear-gradient(135deg, #10b981 0%, #059669 100%);
                        color: white;
                        padding: 8px 16px;
                        text-decoration: none;
                        border-radius: 4px;
                        font-weight: 600;
                        font-size: 13px;
                        transition: all 0.3s ease;
                    " onmouseover="this.style.transform='translateY(-1px)'" onmouseout="this.style.transform='translateY(0)'">
                        🚀 Get PRO Now
                    </a>
                </div>
            </div>
        </div>

        <!-- Upgrade Content -->
        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 30px; margin-top: 20px;">
            
            <!-- Left Column: Features -->
            <div>
                <h2 style="color: #2563eb; margin-bottom: 20px;">PRO Features</h2>
                
                <div style="background: white; border: 1px solid #e2e8f0; border-radius: 8px; padding: 20px;">
                    <h3 style="margin-top: 0; color: #374151;">Advanced Email Alerts</h3>
                    <ul style="color: #6b7280; line-height: 1.6;">
                        <li>HTML email templates with professional design</li>
                        <li>Customizable notification settings</li>
                        <li>Email scheduling and frequency control</li>
                        <li>Multiple recipient support</li>
                    </ul>
                </div>

                <div style="background: white; border: 1px solid #e2e8f0; border-radius: 8px; padding: 20px; margin-top: 15px;">
                    <h3 style="margin-top: 0; color: #374151;">Auto-Rules Engine</h3>
                    <ul style="color: #6b7280; line-height: 1.6;">
                        <li>Automated pingback processing rules</li>
                        <li>Custom filtering and categorization</li>
                        <li>Automatic spam detection and handling</li>
                        <li>Rule-based notifications</li>
                    </ul>
                </div>

                <div style="background: white; border: 1px solid #e2e8f0; border-radius: 8px; padding: 20px; margin-top: 15px;">
                    <h3 style="margin-top: 0; color: #374151;">Analytics & Reporting</h3>
                    <ul style="color: #6b7280; line-height: 1.6;">
                        <li>Comprehensive analytics dashboard</li>
                        <li>Advanced CSV export with custom fields</li>
                        <li>Performance metrics and insights</li>
                        <li>Historical data tracking</li>
                    </ul>
                </div>
            </div>

            <!-- Right Column: Pricing & CTA -->
            <div>
                <div style="background: linear-gradient(135deg, #2563eb 0%, #1e40af 100%); border-radius: 8px; padding: 30px; color: white; text-align: center;">
                    <h2 style="margin: 0 0 10px 0; color: white; font-size: 24px;">
                        <span style="background: #10b981; color: white; padding: 4px 8px; border-radius: 4px; font-size: 12px; margin-right: 8px;">PRO</span>
                        PingBack Source Reveal PRO
                    </h2>
                    
                    <div style="background: rgba(255,255,255,0.1); padding: 25px; border-radius: 8px; margin: 20px 0;">
                        <div style="font-size: 36px; font-weight: bold; color: #10b981; margin-bottom: 5px;">$29</div>
                        <div style="font-size: 16px; opacity: 0.9; margin-bottom: 5px;">One-time payment</div>
                        <div style="font-size: 14px; opacity: 0.8;">Lifetime updates & support</div>
                    </div>

                    <div style="margin: 20px 0;">
                        <a href="https://netclick.tech/pro" target="_blank" style="
                            display: inline-block;
                            background: linear-gradient(135deg, #10b981 0%, #059669 100%);
                            color: white !important;
                            padding: 15px 30px;
                            text-decoration: none;
                            border-radius: 6px;
                            font-weight: 600;
                            font-size: 16px;
                            transition: all 0.3s ease;
                            box-shadow: 0 4px 15px rgba(16, 185, 129, 0.3);
                        " onmouseover="this.style.transform='translateY(-2px)';this.style.boxShadow='0 6px 20px rgba(16, 185, 129, 0.4)'" onmouseout="this.style.transform='translateY(0)';this.style.boxShadow='0 4px 15px rgba(16, 185, 129, 0.3)'">
                            🚀 Get PRO Now
                        </a>
                    </div>

                    <div style="font-size: 13px; opacity: 0.8; margin-top: 15px;">
                        ✓ Instant download after payment<br>
                        ✓ 30-day money-back guarantee<br>
                        ✓ Lifetime updates included
                    </div>
                </div>

                <!-- Additional Info -->
                <div style="background: white; border: 1px solid #e2e8f0; border-radius: 8px; padding: 20px; margin-top: 20px;">
                    <h3 style="margin-top: 0; color: #374151;">What's Included</h3>
                    <ul style="color: #6b7280; line-height: 1.6;">
                        <li>Full PRO plugin download</li>
                        <li>License key for activation</li>
                        <li>Premium support</li>
                        <li>Future updates and new features</li>
                        <li>Documentation and tutorials</li>
                    </ul>
                </div>
            </div>
        </div>
    </div>
    <?php
}

/* ============================================================================
 * (Optional) Keep a small column in the native Comments list for quick glance
 * ==========================================================================*/
add_filter('manage_edit-comments_columns', function ($cols) {
    $cols['pb_source'] = __('Pingback Source','pbr');
    return $cols;
});
add_action('manage_comments_custom_column', function ($col, $id) {
    if ($col !== 'pb_source') return;
    $c = get_comment($id);
    if (!pbr_is_pingtype($c)) { echo '—'; return; }
    $url  = get_comment_meta($id, 'pb_final_url', true);
    if (!$url) $url = get_comment_meta($id, 'pb_source_url', true) ?: $c->comment_author_url;
    $host = get_comment_meta($id, 'pb_source_host', true) ?: ($url ? wp_parse_url($url, PHP_URL_HOST) : '');
    $root = get_comment_meta($id, 'pb_root_domain', true) ?: ($host ? pbr_registrable_domain($host) : '');
    echo $url ? sprintf('<a href="%s" target="_blank" rel="noopener">%s</a>', esc_url($url), esc_html($root ?: $host ?: $url)) : 'Unknown';
}, 10, 2);
